/*-----------------------------------------------------------------------

                         SYRTHES version 3.4
                         -------------------

     This file is part of the SYRTHES Kernel, element of the
     thermal code SYRTHES.

     Copyright (C) 1988-2008 EDF S.A., France

     contact: syrthes-support@edf.fr


     The SYRTHES Kernel is free software; you can redistribute it
     and/or modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2 of
     the License, or (at your option) any later version.

     The SYRTHES Kernel is distributed in the hope that it will be
     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.


     You should have received a copy of the GNU General Public License
     along with the Code_Saturne Kernel; if not, write to the
     Free Software Foundation, Inc.,
     51 Franklin St, Fifth Floor,
     Boston, MA  02110-1301  USA

-----------------------------------------------------------------------*/
# include <stdio.h>
# include <stdlib.h>
# include <math.h>

# include "abs.h"

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | seg_rectanx                                                          |
  |         tester l'intersection entre un segment et un segment x=cte   |
  |======================================================================| */
int seg_rectanx(double dx,double dy,double xa,double xb,double ya,double yb)
{
    double t,d,y;

    d=xb-xa;
    if (abs(d)<1.e-8)
      return(0);
    else
      {
        t=(dx-xa)/d; y=ya+t*(yb-ya);
	if (-dy<=y && y<=dy)
          return(1);
        else 
	  return(0);
      }
}
/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | seg_rectany                                                          |
  |         tester l'intersection entre un segment et un segment y=cte   |
  |======================================================================| */
int seg_rectany(double dx,double dy,double xa,double xb,double ya,double yb)
{
    double t,d,x;

    d=yb-ya;
    if (abs(d)<1.e-8)
      return(0);
    else
      {
        t=(dy-ya)/d; x=xa+t*(xb-xa);
	if (-dx<=x && x<=dx)
          return(1);
        else 
	  return(0);
      }
}

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | seg_cubex                                                            |
  |         tester l'intersection entre une face x=cte                   |
  |======================================================================| */
int seg_cubex(double dx,double dy,double dz,
	      double xa,double xb,double ya,double yb,double za,double zb)
{
    double t,d,y,z;

    d=xb-xa;
    if (abs(d)<1.e-8)
      return(0);
    else
      {
        t=(dx-xa)/d; y=ya+t*(yb-ya);
	if (-dy<=y && y<=dy)
          {
	     z= za+t*(zb-za);
	     if (-dz<=z && z<=dz)
		 return(1);
	     else
		 return(0);
	  }
        else
        return(0);
      }
}
/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | seg_cubey                                                            |
  |         tester l'intersection entre une face y=cte                   |
  |======================================================================| */
int seg_cubey(double dx,double dy,double dz,
	      double xa,double xb,double ya,double yb,double za,double zb)
{
    double t,d,x,z;

    d=yb-ya;
    if (abs(d)<1.e-8)
      return(0);
    else
      {
        t=(dy-ya)/d; x=xa+t*(xb-xa);
	if (-dx<=x && x<=dx)
          {
	     z= za+t*(zb-za);
	     if (-dz<=z && z<=dz)
		 return(1);
	     else
		 return(0);
	  }
        else
        return(0);
      }
}
/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | seg_cubez                                                            |
  |         tester l'intersection entre une face z=cte                   |
  |======================================================================| */
int seg_cubez(double dx,double dy,double dz,
	      double xa,double xb,double ya,double yb,double za,double zb)
{
    double t,d,x,y;

    d=zb-za;
    if (abs(d)<1.e-8)
      return(0);
    else
      {
        t=(dz-za)/d; y=ya+t*(yb-ya);
	if (-dy<=y && y<=dy)
          {
	     x= xa+t*(xb-xa);
	     if (-dx<=x && x<=dx)
		 return(1);
	     else
		 return(0);
	  }
        else
        return(0);
      }
}


/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | diag_tria                                                            |
  |         intersection d'une diagonale de cube avec un plan            |
  |======================================================================| */
int diag_tria(double ta, double tb, double tc, double td, 
	      double dx, double dy, double dz,double *t)
{
    double d,epsi;

    epsi =1.e-8;

    d=ta*dx+tb*dy+tc*dz;

    if (abs(d)<epsi)
      return(0);
    else
      {
	*t= -td/d;
	if (-1<=*t && *t<=1) 
	  return(1);
	else
	  return(0);
      }
}

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_triangle                                                          |
  |         voir si in point est dans un triangle                        |
  |======================================================================| */
int in_triangle (double a,  double b,  double c, double d,
                 double xa, double ya, double za,
                 double xab,double yab,double zab,
                 double xac,double yac,double zac,
                 double xp, double yp, double zp)
{
    int i1;
    double epsi;
    double coo_min,alpha,beta,u0,u1,u2,v0,v1,v2;

    epsi = 1.E-6;

    i1 = 1; coo_min=abs(a);
    if (abs(b)>coo_min) {i1=2;coo_min=abs(b);}
    if (abs(c)>coo_min) {i1=3;}

    switch (i1)
        {
        case 1 : u0=yp-ya; u1=yab; u2=yac;
                 v0=zp-za; v1=zab; v2=zac;
                 break;

        case 2 : u0=xp-xa; u1=xab; u2=xac;
                 v0=zp-za; v1=zab; v2=zac;
                 break;

        case 3 : u0=xp-xa; u1=xab; u2=xac;
                 v0=yp-ya; v1=yab; v2=yac;
                 break;
        }

    alpha = beta = -1;
    if (abs(u1)<epsi)
        {
            beta = u0/u2;
            if (abs(beta)<epsi) beta=0.;
            else if (abs(beta-1.)<epsi) beta=1.;
            if (beta>=0 && beta<=1) alpha = (v0-beta*v2)/v1;
        }
    else
        {
            beta = (v0*u1-u0*v1)/(v2*u1-u2*v1);
            if (abs(beta)<epsi) beta=0.;
            else if (abs(beta-1.)<epsi) beta=1.;
            if (beta>=0 && beta<=1) alpha = (u0-beta*u2)/u1;
        }

/*  printf(" >> in_triangle: alpha, beta = %f %f\n", alpha,beta);   */
    if (alpha>=-epsi  && beta>=-epsi && (alpha+beta)<=1.+ 2.*epsi)
        return(1);
    else
        return(0);
}

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_seg                                                               |
  |         voir si in point est dans un segment                         |
  |======================================================================| */
int in_seg (double xa, double ya, double xb, double yb,
	    double xp, double yp)
{
    double epsi,t;

    epsi = 1.E-6;

    if (abs(xb-xa)<epsi)
      t=(yp-ya)/(yb-ya);
    else
      t=(xp-xa)/(xb-xa);

    if (0<=t && t<=1)
      return(1);
    else
      return(0);
}


/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_rectan                                                            |
  |         voir si in point est dans la boite englobante                |
  |======================================================================| */
int in_rectan (double xv, double yv, double xmin,double xmax,
	       double ymin,double ymax)
{
    double epsi;

    epsi = 1.E-6;

    if (   xv>xmin-epsi && xv<xmax+epsi
        && yv>ymin-epsi && yv<ymax+epsi)
        return(1);
    else
        return(0);
}

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_boite                                                             |
  |         voir si in point est dans la boite englobante                |
  |======================================================================| */
int in_boite (double xv, double yv, double zv, double xmin,double xmax,
              double ymin,double ymax, double zmin,double zmax)
{
    double epsi;

    epsi = 1.E-6;

    if (   xv>xmin-epsi && xv<xmax+epsi
        && yv>ymin-epsi && yv<ymax+epsi
        && zv>zmin-epsi && zv<zmax+epsi )
        return(1);
    else
        return(0);
}


/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_tria_2d                                                           |
  |         voir si in point est dans un triangle en dimension 2         |
  |======================================================================| */
int in_tria_2d (double xa,double ya,double xb,double yb,double xc,double yc,
		double x,double y)
{
    double epsi;
    double alpha,beta,u0,u1,u2,v0,v1,v2;

    epsi = 1.E-6;

    u0=x-xa; u1=xb-xa; u2=xc-xa;
    v0=y-ya; v1=yb-ya; v2=yc-ya;
    
    alpha = beta = -1;
    if (abs(u1)<epsi)
        {
            beta = u0/u2;
            if (abs(beta)<epsi) beta=0.;
            else if (abs(beta-1.)<epsi) beta=1.;
            if (beta>=0 && beta<=1) alpha = (v0-beta*v2)/v1;
        }
    else
        {
            beta = (v0*u1-u0*v1)/(v2*u1-u2*v1);
            if (abs(beta)<epsi) beta=0.;
            else if (abs(beta-1.)<epsi) beta=1.;
            if (beta>=0 && beta<=1) alpha = (u0-beta*u2)/u1;
        }

    if (alpha>=-epsi  && beta>=-epsi && (alpha+beta)<=1.+ 2.*epsi)
        return(1);
    else
        return(0);
}
/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_tria_2d                                                           |
  |         voir si in point est dans un triangle en dimension 2         |
  |======================================================================| */
int in_tria_2d_norm (double a1,double b1,double c1,double a2,double b2,double c2,
		     double a3,double b3,double c3,double x,double y)
{
    double epsi;
    epsi = -1.E-8;

    if ( a1*x+b1*y+c1>epsi &&  a2*x+b2*y+c2>epsi &&  a3*x+b3*y+c3>epsi)
      return(1);
    else
      return(0);
}

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_tetra                                                             |
  |         voir si in point est dans un tetraedre                       |
  |======================================================================| */
int in_tetra_norm (double a1,double b1,double c1,double d1,
		   double a2,double b2,double c2,double d2,
		   double a3,double b3,double c3,double d3,
		   double a4,double b4,double c4,double d4,
		   double x,double y,double z)
{
    double epsi;
    epsi = -1.E-8;

    if (a1*x+b1*y+c1*z+d1>epsi &&  a2*x+b2*y+c2*z+d2>epsi && 
	a3*x+b3*y+c3*z+d3>epsi &&  a4*x+b4*y+c4*z+d4>epsi )
      return(1);
    else
      return(0);
  }

/*|======================================================================|
  | SYRTHES 3.4.3                                     COPYRIGHT EDF 2008 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  | in_tetra                                                             |
  |         voir si in point est dans un tetraedre                       |
  |======================================================================| */
int in_tetra (double xa,double ya,double za,double xb,double yb,double zb,
	      double xc,double yc,double zc,double xd,double yd,double zd,
		double x,double y,double z)
{
    double epsi;
    double alpha,beta,gama,delta,det;
    double xab,yab,zab,xac,yac,zac,xad,yad,zad,xap,yap,zap;
    double xba,yba,zba,xbc,ybc,zbc,xbd,ybd,zbd,xbp,ybp,zbp;
      
    epsi = 1.E-8;

    xab=xb-xa; yab=yb-ya; zab=zb-za;
    xac=xc-xa; yac=yc-ya; zac=zc-za;
    xad=xd-xa; yad=yd-ya; zad=zd-za;
    xap=x -xa; yap=y -ya; zap=z -za;
    det = -xab*zac*yad+xab*yac*zad-yac*zab*xad-yab*xac*zad+zac*yab*xad+zab*xac*yad;
    if (abs(det)>epsi)
      {
	beta = (-xac*yap*zad+xac*yad*zap-xad*yac*zap+xad*zac*yap-xap*zac*yad+xap*yac*zad)/det;
	if (abs(beta)<epsi) beta=0.;
	else if (abs(beta-1.)<epsi) beta=1.;
	if (beta<0 || beta>1) 
	  return(0);
	else
	  {
	    gama = -(-xab*yap*zad+xab*yad*zap-yab*xad*zap-yad*zab*xap+yap*zab*xad+yab*xap*zad)/det;
	    if (abs(gama)<epsi) gama=0.;
	    else if (abs(gama-1.)<epsi) gama=1.;
	    if (gama<0 || gama>1) 
	      return(0);
	    else
	      {
		delta = (xab*yac*zap+zab*xac*yap-yac*zab*xap-xab*zac*yap-yab*xac*zap+zac*yab*xap)/det;
		if (abs(delta)<epsi) delta=0.;
		else if (abs(delta-1.)<epsi) delta=1.;
		if (delta<0 || delta>1) 
		  return(0);
		else
		  {
		    if (abs(xab)>epsi)
		      alpha = -( x-xb - gama*(xc-xb)-delta*(xd-xb) )/xab;
		    else if (abs(yab)>epsi)
		      alpha = -( y-yb - gama*(yc-yb)-delta*(yd-yb) )/yab;
		    else 
		      alpha = -( z-zb - gama*(zc-zb)-delta*(zd-zb) )/zab;
		    if (abs(alpha)<epsi) alpha=0.;
		    else if (abs(alpha-1.)<epsi) alpha=1.;
		    if (alpha<0 || alpha>1)
		      return(0);
		    else if (alpha+beta+gama+delta<=1.+4*epsi) 
		      return(1);
		    else
		      return(0);
		  }
	      }
	  }
      }
    else
      {
	xba=xa-xb; yba=ya-yb; zba=za-zb;
	xbc=xc-xb; ybc=yc-yb; zbc=zc-zb;
	xbd=xd-xb; ybd=yd-yb; zbd=zd-zb;
	xbp=x -xb; ybp=y -yb; zbp=z -zb;

	det = xba*ybc*zbd-xba*zbc*ybd-yba*xbc*zbd+zbc*yba*xbd-ybc*zba*xbd+zba*xbc*ybd;

	alpha = (xbc*ybd*zbp-xbc*zbd*ybp+zbc*xbd*ybp-ybc*xbd*zbp+zbd*ybc*xbp-ybd*zbc*xbp)/det;
	if (abs(alpha)<epsi) alpha=0.;
	else if (abs(alpha-1.)<epsi) alpha=1.;
	if (alpha<0 || alpha>1) 
	  return(0);
	else
	  {
	    gama=-(xba*ybd*zbp-xba*zbd*ybp-yba*xbd*zbp-ybd*zba*xbp+ybp*zba*xbd+yba*xbp*zbd)/det;
	    if (abs(gama)<epsi) gama=0.;
	    else if (abs(gama-1.)<epsi) gama=1.;
	    if (gama<0 || gama>1) 
	      return(0);
	    else
	      {
		delta=(zbc*yba*xbp-xba*zbc*ybp+xba*ybc*zbp-ybc*zba*xbp-yba*xbc*zbp+zba*xbc*ybp)/det;
		if (abs(delta)<epsi) delta=0.;
		else if (abs(delta-1.)<epsi) delta=1.;
		if (delta<0 || delta>1) 
		  return(0);
		else
		  {
		    if (abs(xba)>epsi)
		      beta = -( x-xa - gama*(xc-xa)-delta*(xd-xa) )/xba;
		    else if (abs(yba)>epsi)
		      beta = -( y-ya - gama*(yc-ya)-delta*(yd-ya) )/yba;
		    else 
		      beta = -( z-za - gama*(zc-za)-delta*(zd-za) )/zba;
		    if (abs(beta)<epsi) beta=0.;
		    else if (abs(beta-1.)<epsi) beta=1.;
		    if (beta<0 || beta>1)
		      return(0);
		    else if (alpha+beta+gama+delta<=1.+4*epsi) 
		      return(1);
		    else
		      return(0);
		  }
	      }
	  }
      }

}


