use ui;
use layout;
use graphics;
use core:geometry;
use core:io;

class MainPainter extends Painter {
	// The workspace we want to draw.
	private Workspace workspace;

	// The program we're currently executing.
	private Program? program;

	// Saved value of global variables.
	private SavedGlobals savedGlobals;

	// Problem panel, if any.
	private ProblemPanel? panel;

	// Left mouse button pressed?
	private Bool mouseDown;

	// Current drawable object in focus. Only useful when 'mousedown' is pressed.
	private Drawable? mouseFocus;

	// Location mouse was pressed at.
	private Point mouseOffset;

	// Offset for the workspace.
	private Point offset;

	// Scale factor for the workspace. Applied last, meaning that "offset" and "mouseOffset" are also scaled.
	private Float scale;

	// Behavior to use in the painter.
	private Behavior? behavior;

	// Lock for traversal. The traversal may perform thread switches, so we need to make sure that
	// multiple traversals are not in flight at the same time.
	private sync:Lock traversalLock;

	init() {
		init() { scale = 1.0; }
		bgColor = Color(0.8, 0.8, 0.8);
	}

	// Set whether or not to track reads/writes.
	void trackMemory(Bool track) {
		workspace.trackMemory = track;
	}

	// Set zoom. Also pans to the center to make sure nothing gets hidden too badly.
	void setZoom(Float zoom) {
		if (scale != zoom) {
			offset = Point();
			scale = zoom;
		}
	}

	// Do we have an open "program" instance?
	Bool hasProgram() {
		program.any;
	}

	// Update the behavior without re-loading a program.
	void updateBehavior(Behavior b) {
		behavior = b;
		panel = b.panel;
		workspace.enableStep = b.allowStep;
	}

	// Open a program.
	void open(Url[] files, Behavior b) {
		var loaded = Program:load(files);

		// Terminate the previous program, if any. We do that here in case the load failed.
		cleanup();

		program = loaded;
		updateBehavior(b);
		var globals = loaded.findGlobals();
		savedGlobals = SavedGlobals:save(globals);
		workspace.globals = globals;
		workspace.hints = loaded.hints;
		workspace.trackDeadlock = true;
		loaded.onChange = &this.update();
		loaded.onError = &this.onProgramError;
		loaded.onAdvance = &this.onProgramAdvance;
		update();
		loaded.spawn(loaded.main);
	}

	// Try to find the loaded source code for the program.
	Str? sourceFor(Url url) {
		if (program)
			if (text = program.source(url))
				return text.text;
		null;
	}

	// Apply an action. These are the same that we produce to the Behavior.
	void applyAction(Str action) {
		unless (program)
			return;

		if (action == "s") {
			// Spawn new thread.
			spawnThread();
		} else if (action.isNat) {
			// Step a thread.
			Nat thread = action.toNat;
			for (v in program.threads)
				if (v.threadId == thread)
					v.resume();
		}
	}

	private void onProgramError(Nat thread, Exception error) {
		unless (behavior)
			return;

		// These are ones we can dispatch directly.
		Str type = "exception";
		Str msg = error.message;

		// See if it is an error we recognize.
		if (error as TypedError)
			type = error.type;

		stopThreads();
		behavior.onProgramError(ThreadError(type, msg, thread));
	}

	private void onProgramAdvance(Nat thread) {
		unless (behavior)
			return;

		behavior.onUserAction(thread.toS);
	}

	// Clean up the running program (i.e. make sure all threads are terminated).
	void cleanup() {
		// Don't call any behavior during termination.
		behavior = null;
		panel = null;

		if (program)
			program.terminate();

		workspace.clear();
		offset = Point();
		repaint();
	}

	// Restart the program.
	void restart() {
		if (program) {
			// Don't call behavior during restart.
			var tmp = behavior;
			behavior = null;

			// TODO: We need to re-initialize globals as well!

			program.terminate();
			workspace.clear();
			var globals = program.findGlobals();
			savedGlobals.restore(globals);
			workspace.globals = globals;
			workspace.trackDeadlock = true;

			behavior = tmp;
			if (behavior)
				behavior.onRestart();
			program.spawn(program.main);
		}
	}

	// Spawn another thread.
	void spawnThread() {
		if (program) {
			program.spawn(program.main);
			if (behavior)
				behavior.onUserAction("s");
		}
	}

	// Stop all running threads.
	void stopThreads() {
		if (program) {
			for (k, v in program.threads) {
				v.pause();
			}
		}
	}

	// Start all threads with the specified speed
	void resumeThreads(Duration speed) {
		if (program) {
			for (k, v in program.threads) {
				v.resume(speed);
			}
		}
	}

	// Render everything.
	Bool render(Size me, Graphics g) : override {
		g.transform(translate(-offset) * scale(scale));

		Rect viewport(offset, me / scale);
		workspace.render(viewport, g);

		if (panel)
			panel.render(viewport, g);

		false;
	}

	// Update the visualization of the data.
	void update() {
		sync:Lock:Guard guard(traversalLock);
		workspace.beginTraversal();

		if (program) {
			for (id, thread in program.threads) {
				workspace.traverse(thread);
			}

			if (output = program.getOutput()) {
				workspace.outputText(output);
			}
		}

		var error = workspace.endTraversal();

		repaint();

		if (error) {
			if (behavior) {
				stopThreads();
				behavior.onProgramError(error);
			}
		}
	}

	// Called when the mouse is moved.
	void mouseMoved(Point pt) {
		if (!mouseDown)
			return;

		pt /= scale;
		pt += offset;

		if (mouseFocus) {
			// Call the drawable.
			if (mouseFocus.mouseMoved(pt, mouseOffset))
				repaint();
		} else {
			// Update our transform.
			offset -= pt - mouseOffset;
			repaint();
		}
	}

	// Called when a mouse button is clicked.
	void mouseClicked(Point pt, Bool down, MouseButton button) {
		pt /= scale;
		pt += offset;

		if (button == MouseButton:left) {
			if (down) {
				if (panel) {
					if (panel.mouseClicked(pt))
						return;
				}

				mouseDown = true;
				mouseOffset = pt;
				mouseFocus = workspace.findAtActivate(pt);

				if (mouseFocus) {
					mouseFocus.mouseClicked(pt, down);
				}

				repaint();
			} else {
				mouseOffset = pt;
				mouseLeave();
			}
		}
	}

	// Called when the mouse leaves the window.
	void mouseLeave() {
		if (mouseFocus) {
			// Tell it we released the cursor. We might want to differentiate between the two cases later on...
			if (mouseFocus.mouseClicked(mouseOffset, false))
				repaint();
		}

		mouseDown = false;
		mouseFocus = null;
	}
}
