/* properties.c - maintain the properties dialog
 *
 * Copyright (C) 2000  Jochen Voss.  */

static const  char  rcsid[] = "$Id: properties.c 5734 2004-06-03 19:47:31Z voss $";

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <time.h>

#include "support.h"

#include "sandcommon.h"
#include "sand-window.h"
#include "locate.h"


void
properties_timer_stopped (GtkWidget *properties)
/* This must be called after the timer elapsed.
 * It disables some of the dialog's fields.  */
{
  gtk_widget_set_sensitive (lookup_widget (properties, "time_entry"), FALSE);
  gtk_widget_set_sensitive (lookup_widget (properties, "msg_entry"), FALSE);

  gtk_widget_set_sensitive (lookup_widget (properties, "bell_alarm"), FALSE);
  gtk_widget_set_sensitive (lookup_widget (properties, "snd_alarm"), FALSE);
  gtk_widget_set_sensitive (lookup_widget (properties, "cmd_alarm"), FALSE);
  gtk_widget_set_sensitive (lookup_widget (properties, "corba_alarm"), FALSE);

  gtk_widget_set_sensitive (lookup_widget (properties, "snd_entry"), FALSE);
  gtk_widget_set_sensitive (lookup_widget (properties, "cmd_entry"), FALSE);
  gtk_widget_set_sensitive (lookup_widget (properties, "corba_entry"), FALSE);
}

static void
prepare_page0 (GtkWidget *properties, const struct timer *timer)
/* Set the dialogs's alarm parameters at page 0.  */
{
  GtkWidget *time_entry;
  char  buffer [32];

  time_entry = lookup_widget (properties, "time_entry");
  {
    time_t target_time_abs = (int)(timer->target_time_abs + 0.5);
    strftime (buffer, 32, "%Y-%m-%d %H:%M:%S", localtime (&target_time_abs));
  }
  gtk_entry_set_text (GTK_ENTRY (time_entry), buffer);

  if (timer->message) {
    GtkWidget *msg_entry = lookup_widget (properties, "msg_entry");
    gtk_entry_set_text (GTK_ENTRY (msg_entry), timer->message);
  }
}

static void
prepare_page1 (GtkWidget *properties, const struct timer *timer)
/* Initialize the dialog's alarm action at page 1.  */
{
  CORBA_Environment  ev;

  CORBA_exception_init (&ev);
  {
    GtkWidget *snd_entry = lookup_widget (properties, "snd_file_entry");
    char *sound_dir = get_sound_dir ();

    gnome_file_entry_set_default_path (GNOME_FILE_ENTRY (snd_entry),
				       sound_dir);
    g_free (sound_dir);
  }

  if (CORBA_Object_is_a (timer->alarm, "IDL:SandUhr/AlarmBeep:1.0", &ev)) {
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (properties, "bell_alarm")), TRUE);
  } else if (CORBA_Object_is_a (timer->alarm,
				"IDL:SandUhr/AlarmSound:1.0", &ev)) {
    GtkWidget *snd_file;
    char *snd;

    snd_file = lookup_widget (properties, "snd_entry");
    snd = SandUhr_AlarmSound__get_SoundFile (timer->alarm, &ev);
    gtk_entry_set_text (GTK_ENTRY (snd_file), snd);
    CORBA_free (snd);
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (properties, "snd_alarm")), TRUE);
  } else if (CORBA_Object_is_a (timer->alarm,
				"IDL:SandUhr/AlarmCommand:1.0", &ev)) {
    GtkWidget *cmd_entry;
    char *cmd;
    
    cmd_entry = lookup_widget (properties, "cmd_entry");
    cmd = SandUhr_AlarmCommand__get_CommandString (timer->alarm, &ev);
    gtk_entry_set_text (GTK_ENTRY (cmd_entry), cmd);
    CORBA_free (cmd);
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (properties, "cmd_alarm")), TRUE);
  } else {
    GtkWidget *corba_entry;
    corba_entry = lookup_widget (properties, "corba_entry");
    gtk_entry_set_text (GTK_ENTRY (corba_entry),
			CORBA_ORB_object_to_string (orb, timer->alarm, &ev));
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (properties, "corba_alarm")), TRUE);
  }
  check_corba_error (&ev, NULL);
}

static void
prepare_page2 (GtkWidget *properties, const struct timer *timer)
/* Initialize the dialog's sand settings at page 2.  */
{
  GtkWidget *color_picker = lookup_widget (properties, "sand_color");
  guint8  r, g, b;

  sand_window_get_color (SAND_WINDOW (timer->window), &r, &g, &b);
  gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (color_picker),
			     r, g, b, 255);
}

static void
prepare_page3 (GtkWidget *properties, const struct timer *timer)
/* Initialize the dialog's window attributes at page 3.  */
{
  GtkWidget *deco_button = lookup_widget (properties, "deco_button");
  gboolean  decorations;

  g_object_get (G_OBJECT (timer->window),
		SAND_WINDOW_GET_PROP_DECORATIONS (&decorations),
		NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (deco_button),
				decorations);
}

void
prepare_properties_dialog (GtkWidget *properties, struct timer *timer)
{
  prepare_page0 (properties, timer);
  prepare_page1 (properties, timer);
  prepare_page2 (properties, timer);
  prepare_page3 (properties, timer);
  if (timer->state == SandUhr_Timer_TSDone)
    properties_timer_stopped (properties);
}

static void
page0_to_timer (GtkWidget *properties, struct timer *timer)
/* Set the timer's alarm parameters from page 0 of the dialog.  */
{
  GtkWidget *time_entry = lookup_widget (properties, "time_entry");
  GtkWidget *msg_entry = lookup_widget (properties, "msg_entry");
  const char *timestr, *msg;
  
  timestr = gtk_entry_get_text (GTK_ENTRY (time_entry));
  begin_arg (timestr);
  if (yyparse (timer) == 0) {
    initialize_time (timer);
  } else {
    gchar *msg;
    msg = g_strdup_printf (_("invalid time specification\n"
			     "\"%s\""), timestr);
    display_error_message (msg, GTK_WINDOW(properties));
    g_free (msg);
  }
  end_arg ();
      
  g_free (timer->message);
  msg = gtk_entry_get_text (GTK_ENTRY (msg_entry));
  if (msg && *msg) {
    timer->message = g_strdup (msg);
    gnome_config_set_string ("/SandUhr/preferences/message", msg);
  } else {
    timer->message = NULL;
  }
}

static void
page1_to_timer (GtkWidget *properties, struct timer *timer)
/* Set the timer's alarm action from page 1 of the dialog.  */
{
  CORBA_Environment  ev;
  SandUhr_AlarmAction  new_alarm = NULL;
  SandUhr_Timer timer_ref;
 
  CORBA_exception_init (&ev);
  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (lookup_widget (properties, "bell_alarm")))) {
    struct alarm_beep *ab;
    
    ab = create_alarm_beep (timer->poa, 3, &ev);
    new_alarm = PortableServer_POA_servant_to_reference (timer->poa, ab, &ev);
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (properties, "snd_alarm")))) {
    GtkWidget *snd_entry = lookup_widget (properties, "snd_entry");
    const char *file = gtk_entry_get_text (GTK_ENTRY (snd_entry));
    struct alarm_sound *as;

    as = create_alarm_sound (timer->poa, file, &ev);
    new_alarm = PortableServer_POA_servant_to_reference (timer->poa, as, &ev);
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (properties, "cmd_alarm")))) {
    GtkWidget *cmd_entry = lookup_widget (properties, "cmd_entry");
    const char *cmd = gtk_entry_get_text (GTK_ENTRY (cmd_entry));
    struct alarm_command *ac;

    ac = create_alarm_command (timer->poa, cmd, &ev);
    new_alarm = PortableServer_POA_servant_to_reference (timer->poa, ac, &ev);
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (properties, "corba_alarm")))) {
    GtkWidget *corba_entry = lookup_widget (properties, "corba_entry");
    const char *ior = gtk_entry_get_text (GTK_ENTRY (corba_entry));

    new_alarm = CORBA_ORB_string_to_object (orb, ior, &ev);
  } else {
    g_assert_not_reached ();
  }
  check_corba_error (&ev, GTK_WINDOW(properties));

  if (! new_alarm)  return;
  timer_ref = PortableServer_POA_servant_to_reference (timer->poa, timer, &ev);
  SandUhr_Timer__set_Alarm (timer_ref, new_alarm, &ev);
  CORBA_Object_release (timer_ref, &ev);
  check_corba_error (&ev, GTK_WINDOW(properties));
}

static void
page2_to_timer (GtkWidget *properties, struct timer *timer)
/* Set the timer's sand settings from page 2 of the dialog.  */
{
  GtkWidget *color_picker = lookup_widget (properties, "sand_color");
  guint8  r, g, b, a;

  gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (color_picker),
			     &r, &g, &b, &a);
  sand_window_set_color (SAND_WINDOW (timer->window), r, g, b);
}

static void
page3_to_timer (GtkWidget *properties, struct timer *timer)
/* Set the timer's window attributes from page 3 of the dialog.  */
{
  GtkWidget *deco_button = lookup_widget (properties, "deco_button");
  gboolean  decorations;

  decorations
    = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (deco_button));
  g_object_set (G_OBJECT (timer->window),
		SAND_WINDOW_PROP_DECORATIONS (decorations),
		NULL);
}

void
properties_to_timer (GtkWidget *properties, int page, struct timer *timer)
{

  switch (page) {
  case 0:
    page0_to_timer (properties, timer);
    break;
  case 1:
    page1_to_timer (properties, timer);
    break;
  case 2:
    page2_to_timer (properties, timer);
    break;
  case 3:
    page3_to_timer (properties, timer);
    break;
  default:
    g_assert (page == -1);
    factory_update_timer (timer->factory, timer);
    gnome_config_sync ();
    break;
  }
}
